function [x,psi,rho]=psirho(obj,pas,xmin,xmax)
% PSIRHO calcule les fonctions d'onde et les densits de probabilit
%   associes  un objet physique interagissant avec une marche d'nergie
%   potentielle. Le calcul s'effectue sur un domaine de l'axe des x
%   compris entre xmin et xmax.
%   [x,psi,rho]=PSIRHO(obj,pas,xmin,xmax)
%       >> obj  = objet informatique marche;
%       >> pas  = incrment selon l'axe x;
%       >> xmin = limite infrieure de l'axe x;
%       >> xmax = limite suprieure de l'axe x;
%       << x    = variable de position sur l'axe x;
%       << psi  = fonction d'onde de l'objet physique.
%       << rho  = densit de probabilit de l'objet physique.
%
%   Olivier Pujol : septembre 2013
%

if (isa(obj,'marche')==0),error('Mauvais type d''objet !');end
if (pas>=0.01),error('Le pas pourrait tre trop grand !');end
if ( (xmin>=obj.x)||(xmax<=obj.x) ),error('Objet marche hors du domaine !');end
%
mat=matrix(obj);
%
x=cell(1,2);
psi=cell(1,2);
rho=cell(1,2);

%   Entre
x{1}=linspace(xmin,obj.x,1+(obj.x-xmin)/pas);
psi{1}=nan(size(x{1}));
psi{1}=(1/sqrt(obj.ke))*(exp(i*obj.ke*(x{1}-obj.x)) ...
                        + (-mat(2,1)/mat(2,2))*exp(-i*obj.ke*(x{1}-obj.x)) );
rho{1}=nan(size(x{1}));
rho{1}=psi{1}.*conj(psi{1});
%
%   Sortie
x{2}=linspace(obj.x,xmax,1+(xmax-obj.x)/pas);
psi{2}=nan(size(x{2}));
psi{2}=(1/sqrt(obj.ks))*(1/mat(2,2))*exp(i*obj.ks*(x{2}-obj.x));
rho{2}=nan(size(x{2}));
rho{2}=psi{2}.*conj(psi{2});
%
% Normalisation de psi et rho : on dtermine l'aire sous la courbe rho(x) entre
% xmin et xmax.
aire = 0.5*sum((rho{1}(1:end-1)+rho{1}(2:end))*pas) + 0.5*sum((rho{2}(1:end-1)+rho{2}(2:end))*pas)/2;
rho{1}=rho{1}/aire;
rho{2}=rho{2}/aire;
%
psi{1}=psi{1}/sqrt(aire);
psi{2}=psi{2}/sqrt(aire);
%
return;